<?php

/**
 * Exports selected snippets to a XML or PHP file.
 *
 * It's better to call the export_snippets()
 * function than directly using this class
 *
 * @since 1.9
 * @package Code_Snippets
 */
class Code_Snippets_Export {

	/**
	 * The IDs
	 * @var array
	 */
	protected $snippet_ids = array();

	/**
	 * The name of the table to fetch snippets from
	 * @var string
	 */
	protected $table_name = '';

	/**
	 * The export file format.
	 * Either 'xml' or 'php'
	 * @var string
	 */
	protected $format;

	/**
	 * The DOM document.
	 * Only used in XML exports
	 * @var DOMDocument
	 */
	protected $dom;

	/**
	 * The DOM document root element
	 * Only used in XML exports
	 * @var DOMElement
	 */
	protected $root;

	/**
	 * Constructor function
	 *
	 * @param array  $ids    The IDs of the snippets to export
	 * @param string $table  The name of the table to fetch snippets from
	 * @param string $format The format of the export file
	 */
	public function __construct( $ids, $table, $format = 'xml' ) {
		$this->snippet_ids = (array) $ids;
		$this->table_name = $table;
		$this->format = 'php' === $format ? 'php' : 'xml';
	}

	/**
	 * Build the export file name
	 * @return string
	 */
	public function get_filename() {

		if ( 1 == count( $this->snippet_ids ) ) {
			/* If there is only snippet to export, use its name instead of the site name */
			$snippet = get_snippet( $this->snippet_ids[0], $this->table_name );
			$title = strtolower( $snippet->name );
		} else {
			/* Otherwise, use the site name as set in Settings > General */
			$title = strtolower( get_bloginfo( 'name' ) );
		}

		$filename = "{$title}.code-snippets.{$this->format}";
		$filename = apply_filters( 'code_snippets/export/filename', $filename, $title );
		return sanitize_file_name( $filename );
	}

	/**
	 * Append header comments to the DOM document
	 */
	protected function do_header_comments() {

		/* Array of translated comment lines */
		$lines = array(
			__( 'This is a code snippets export file generated by the Code Snippets WordPress plugin.', 'code-snippets' ),
			'https://wordpress.org/plugins/code-snippets',
			__( 'To import these snippets a WordPress site follow these steps:', 'code-snippets' ),
			__( '1. Log in to that site as an administrator.', 'code-snippets' ),
			__( '2. Install the Code Snippets plugin using the directions provided at the above link.', 'code-snippets' ),
			__( "3. Go to 'Tools: Import' in the WordPress admin panel.", 'code-snippets' ),
			__( '4. Click on the "Code Snippets" importer in the list', 'code-snippets' ),
			__( '5. Upload this file using the form provided on that page.', 'code-snippets' ),
			__( '6. Code Snippets will then import all of the snippets and associated information contained in this file into your site.', 'code-snippets' ),
			__( "7. You will then have to visit the 'Snippets: All Snippets' admin menu and activate desired snippets.", 'code-snippets' ),
		);

		/* Add each line as a comment element */
		foreach ( $lines as $line ) {
			$comment = $this->dom->createComment( " $line " );
			$this->dom->appendChild( $comment );
		}

		/* Build a generator line, like the WordPress export files */
		$gen = sprintf(
			'generator="Code Snippets/%s" created="%s"',
			CODE_SNIPPETS_VERSION,
			date( 'Y-m-d H:i' )
		);

		/* Run the generator line through the standard WordPress filter */
		$type = 'code_snippets_export';
		$gen = apply_filters( "get_the_generator_$type", $gen, $type );

		/* Add it to the file as a comment */
		$gen = $this->dom->createComment( " $gen " );
		$this->dom->appendChild( $gen );
	}

	/**
	 * Process all snippet items
	 */
	protected function do_items() {
		global $wpdb;

		if ( 'xml' === $this->format ) {

			/* Create the root element */
			$root_element = $this->dom->createElement( 'snippets' );
			$this->root = $this->dom->appendChild( $root_element );
		}

		/* Loop through the snippets */
		foreach ( $this->snippet_ids as $id ) {

			/* Grab the snippet from the database */
			$snippet = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$this->table_name} WHERE id = %d", $id ), ARRAY_A );
			$snippet = new Snippet( $snippet );

			/* Process the snippet item */
			if ( 'php' === $this->format ) {
				$this->do_item_php( $snippet );
			} else {
				$this->do_item( $snippet );
			}
		}
	}

	/**
	 * Append a single snippet item to the document
	 * @param Snippet $snippet
	 */
	protected function do_item( Snippet $snippet ) {
		$item_element = $this->dom->createElement( 'snippet' );
		$item = $this->root->appendChild( $item_element );

		/* Set the scope attribute */
		/** @var $item DOMElement */
		$item->setAttribute( 'scope', $snippet->scope );

		/* Fields to include in the export file */
		$fields = array(
			'name' => $snippet->name,
			'desc' => $snippet->desc,
			'tags' => $snippet->tags_list,
			'code' => $snippet->code,
		);

		$fields = apply_filters( 'code_snippets/export/xml_fields', $fields, $snippet );

		foreach ( $fields as $field_name => $field_value ) {

			/* Create a new element for each field */
			$field_element = $this->dom->createElement( $field_name );
			$field = $item->appendChild( $field_element );

			/* Add the field content */
			$value = $this->dom->createTextNode( $field_value );
			$field->appendChild( $value );
		}
	}

	/**
	 * Format single snippet item as PHP code
	 * @param Snippet $snippet
	 */
	protected function do_item_php( Snippet $snippet ) {
		echo "\n/**\n * {$snippet->name}\n";

		if ( ! empty( $snippet->desc ) ) {

			/* Convert description to PhpDoc */
			$desc = strip_tags( str_replace( "\n", "\n * ", $snippet->desc ) );

			echo " *\n * $desc\n";
		}

		echo " */\n{$snippet->code}\n";
	}

	/**
	 * Export the snippets
	 */
	public function do_export() {

		/* Make the page act like a downloadable file instead of being shown in the browser */
		$filename = $this->get_filename();
		header( 'Content-Disposition: attachment; filename=' . $filename );

		if ( 'xml' === $this->format ) {

			/* Set the HTTP content type header */
			header( 'Content-Type: text/xml; charset=' . get_bloginfo( 'charset' ) );

			/* Create DOM document and root element */
			$this->dom = new DOMDocument( '1.0', get_bloginfo( 'charset' ) );
			$this->dom->formatOutput = true;

			/* Add file header comments */
			$this->do_header_comments();
		}

		elseif ( 'php' === $this->format ) {
			echo '<?php';
		}

		/* Process the snippet items */
		$this->do_items();

		if ( 'xml' === $this->format ) {

			/* Filter DOM document */
			apply_filters( 'code_snippets/export_dom_document', $this->dom, $this->snippet_ids, $filename );

			/* Send the document to the browser */
			echo $this->dom->saveXML();
		}

		exit;
	}
}
